<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../includes/functions.php';

// Check if user is logged in and has permission
if (!isLoggedIn() || !hasPermission('manage_products')) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

header('Content-Type: application/json');

try {
    // Get JSON data
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!isset($data['receipt_id']) || !isset($data['product_id']) || !isset($data['quantity'])) {
        throw new Exception('Missing required fields');
    }
    
    $receipt_id = intval($data['receipt_id']);
    $product_id = intval($data['product_id']);
    $quantity = intval($data['quantity']);
    
    // Start transaction
    $pdo->beginTransaction();
    
    // Check if receipt exists and is approved
    $stmt = $pdo->prepare("SELECT status FROM supplier_receipts WHERE id = ?");
    $stmt->execute([$receipt_id]);
    $receipt = $stmt->fetch();
    
    if (!$receipt || $receipt['status'] !== 'approved') {
        throw new Exception('Invalid or unapproved receipt');
    }
    
    // Update product quantity
    $stmt = $pdo->prepare("UPDATE products SET quantity = quantity + ? WHERE id = ?");
    $stmt->execute([$quantity, $product_id]);
    
    // Update receipt status to 'completed'
    $stmt = $pdo->prepare("UPDATE supplier_receipts SET status = 'completed' WHERE id = ?");
    $stmt->execute([$receipt_id]);
    
    // Log the adjustment
    $stmt = $pdo->prepare("
        INSERT INTO stock_adjustments 
        (product_id, quantity, reason, adjusted_by, adjusted_at) 
        VALUES (?, ?, ?, ?, NOW())
    ");
    $stmt->execute([
        $product_id, 
        $quantity, 
        'Added from supplier receipt #' . $receipt_id,
        $_SESSION['user_id']
    ]);
    
    // Log to system logs
    $product_stmt = $pdo->prepare("SELECT name FROM products WHERE id = ?");
    $product_stmt->execute([$product_id]);
    $product = $product_stmt->fetch();
    
    $details = sprintf(
        "Product: %s (ID: %d), Added: %d from supplier receipt #%d, By: %s (User ID: %d)",
        $product['name'],
        $product_id,
        $quantity,
        $receipt_id,
        $_SESSION['username'],
        $_SESSION['user_id']
    );
    logSystemAction('stock_added_from_supplier', $details);
    
    // Commit transaction
    $pdo->commit();
    echo json_encode(['success' => true]);
    
} catch (Exception $e) {
    // Rollback transaction on error
    $pdo->rollBack();
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
